# User API Documentation

## Overview
The User API provides endpoints for user authentication, task completion, referral management, daily checkins, and account operations. All requests and responses are encrypted using the app's encryption system.

## Base URL
```
https://quizmate.snaplimit.com/api/user/
```

## Authentication
All authenticated endpoints require:
- **Device-ID**: Required header for all endpoints
- **Session-Token**: Required header for authenticated endpoints (except registration/login)
- **Rate Limit**: 100 requests per hour per user

## Encryption
All requests and responses use a custom encryption system:
- **Request Encryption**: Base64 encoding + vowel replacement obfuscation
- **Response Encryption**: Same encryption applied to JSON responses

## Endpoints

### 1. User Registration/Login
**Endpoint:** `POST /account`

**Description:** Register a new user or login existing user with "one device, one account" policy

**Headers:**
```
Device-ID: <device_id>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "email": "user@example.com",
  "device_id": "device123"
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Login successful",
  "data": {
    "user_id": 1,
    "email": "user@example.com",
    "refercode": "userxIrP"
  },
  "session_token": "d287681ad6cca1c8d4d70dba7173126adbdcc4d9ada9f4da6fde42dc7d15ffcf"
}
```

**Error Responses:**
- 400: Device-ID header required, Empty request body, Invalid request body, Device-ID mismatch, Device already used, Email already registered
- 429: Too many requests
- 500: Unable to generate unique referral code, Signup failed

**Special Features:**
- **One Device Policy**: Each device can only be associated with one account
- **Email Masking**: Error messages show masked email addresses for privacy
- **Auto Referral Code**: Automatically generates unique 8-character referral codes

---

### 2. User Logout
**Endpoint:** `POST /logout`

**Description:** Logout user and invalidate session

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Logout successful"
}
```

**Error Responses:**
- 400: Empty request body, Invalid request body, Missing user_id
- 401: Invalid session or device
- 405: Method not allowed

---

### 3. Delete Account
**Endpoint:** `POST /delete_account`

**Description:** Permanently delete user account and all associated data

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Account deleted successfully"
}
```

**Error Responses:**
- 400: Empty request body, Invalid request body, Missing user_id
- 401: Invalid session or device
- 500: Failed to delete account

**Data Cleanup:**
- Deletes session tokens, redeem history, promo code usage, referral rewards
- Updates referred_by for users who were referred by deleted user

---

### 4. Get App Settings
**Endpoint:** `POST /app_settings`

**Description:** Fetch app configuration and user-specific settings

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "App settings fetched",
  "data": {
    "update": {
      "app_link": "https://example.com/app.apk",
      "version": "1.0.1",
      "version_code": 2,
      "changelogs": "Bug fixes and performance improvements"
    },
    "maintenance": 0,
    "vpn_allowed": 1,
    "is_blocked": 0,
    "links": {
      "how_to": "https://example.com/how-to",
      "telegram": "https://t.me/rewardapp",
      "support": "https://example.com/support"
    }
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid user_id
- 401: Invalid session or device

### 4.1. Check User Block Status
**Endpoint:** `POST /app_settings` (with action = "user_blocked")

**Description:** Check if a user is blocked. Returns only the block status for the user.

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "action": "user_blocked"
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "User block status fetched",
  "data": {
    "is_blocked": 0
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid user_id, Invalid request body, Invalid action
- 401: Invalid session or device
- 404: User not found
- 429: Too many requests
- 500: Database connection failed

---

### 5. Apply Referral Code
**Endpoint:** `POST /apply_refercode`

**Description:** Apply a referral code to earn bonus coins immediately

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "refercode": "userxIrP"
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Referral code applied successfully",
  "data": {
    "referrer_coins_awarded": 50,
    "referred_coins_awarded": 50,
    "message": "Referral code applied successfully! You earned 50 coins and the referrer earned 50 coins."
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid referral code format, Already used referral code, Cannot use own referral code, Invalid referral code
- 404: User not found
- 500: Unable to retrieve referral settings, Database error

**Features:**
- **Immediate Rewards**: No eligibility checks or coin thresholds
- **Self-Referral Prevention**: Users cannot use their own referral codes
- **Format Validation**: 8-character alphanumeric codes only
- **Transaction Safety**: Database transactions ensure data consistency

---

### 6. Complete Task
**Endpoint:** `POST /complete_task`

**Description:** Complete a task and earn coins. When website tasks are enabled, this endpoint assigns website tasks instead of showing ads.

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "earned_coins": 10,
  "by_task": "word_puzzle",
  "created_at": "2025-01-15 10:30:00"
}
```

**Parameters:**
- `user_id` (integer, required): User ID
- `earned_coins` (integer, required): Coins earned from task completion
- `by_task` (string, optional): Task type for limits tracking (word_puzzle, alphabet_puzzle, count_images, big_vs_small). If not provided, task is unlimited.
- `created_at` (string, optional): Task completion timestamp

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Task completed",
  "data": {
    "show_ad": true,
    "total_coins": 160,
    "task_limits": {
      "enabled": true,
      "remaining_limit": 14,
      "daily_limit": 20
    },
    "website_task": {
      "id": 5,
      "url": "https://example.com",
      "timer": 30
    },
    "referral_reward": {
      "awarded": true
    }
  },
  "new_session_token": "new_session_token_here"
}
```

**Website Task Response (when enabled):**
- `show_ad` (boolean): Set to `false` when website task is assigned
- `website_task` (object, optional): Website task details when `website_task_enabled` is true
  - `id` (integer): Website task ID for completion
  - `url` (string): URL to visit
  - `timer` (integer): Required visit duration in seconds

**Task Limits Response:**
- `enabled` (boolean): Whether task limits are enabled for this task
- `remaining_limit` (integer): Remaining attempts for today (null if limits disabled or by_task not provided)
- `daily_limit` (integer): Total daily limit for this task type (null if limits disabled or by_task not provided)

**Error Responses:**
- 400: Empty request body, Invalid request body, Missing user_id
- 401: Invalid session or device
- 500: Database error, Daily limit reached for task type

**Features:**
- **Task Limits**: Daily limits per task type when by_task is provided
- **Unlimited Tasks**: Tasks without by_task parameter have no limits
- **Website Task Integration**: Returns website visit tasks when enabled
- **Referral Rewards**: Immediate referral rewards without coin thresholds
- **Task Completion Tracking**: Records task completions in database
- **Ad Display Control**: Controls whether to show ads or website tasks
- **Backward Compatible**: Works without by_task parameter (no limits applied)

**Website Task Flow:**
1. When `website_task_enabled` is true, this endpoint assigns a website task
2. Client should redirect user to the website URL and track visit duration
3. After visit completion, call `/complete_website_task` endpoint to claim rewards
4. When `website_task_enabled` is false, normal ad flow continues

**Task Limits Configuration:**
- **word_puzzle**: 20 attempts per day
- **alphabet_puzzle**: 15 attempts per day
- **count_images**: 10 attempts per day
- **big_vs_small**: 8 attempts per day
- **website_visit**: Unlimited (no daily limits)

**Usage Examples:**
```json
// With task limits (by_task provided) - Limited to daily quota
{
  "user_id": 1,
  "earned_coins": 10,
  "by_task": "word_puzzle"
}

// Without task limits (no by_task) - Unlimited completions
{
  "user_id": 1,
  "earned_coins": 10
}

// Custom task without limits (by_task not in configured list) - Unlimited
{
  "user_id": 1,
  "earned_coins": 10,
  "by_task": "custom_task"
}
```

**Important Notes:**
- **Tasks without `by_task` parameter are unlimited** and not tracked
- **Tasks with `by_task` not in the configured list are also unlimited**
- **Only tasks with `by_task` matching configured task types are limited**
- **Daily limits reset at midnight each day**
- **Website tasks require separate completion via `/complete_website_task` endpoint**

---

### 7. Complete Website Task
**Endpoint:** `POST /complete_website_task`

**Description:** Complete a website visit task and earn coins. This endpoint is called after the user has visited the assigned website for the required duration.

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "visit_duration": 35,
  "earned_coins": 10,
  "created_at": "2025-01-15 10:30:00"
}
```

**Parameters:**
- `user_id` (integer, required): User ID
- `visit_duration` (integer, required): Actual time spent on website in seconds
- `earned_coins` (integer, required): Coins earned from website task completion
- `created_at` (string, optional): Task completion timestamp

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Website task completed successfully",
  "data": {
    "show_ad": false,
    "total_coins": 170,
    "earned_coins": 10,
    "task_limits": {
      "enabled": false,
      "remaining_limit": null,
      "daily_limit": null
    },
    "website_task": {
      "id": 5,
      "url": "https://example.com",
      "timer": 30,
      "visit_duration": 35
    },
    "referral_reward": {
      "awarded": false
    }
  },
  "new_session_token": "new_session_token_here"
}
```

**Response Data:**
- `show_ad` (boolean): Always false for website task completion
- `total_coins` (integer): User's updated coin balance
- `earned_coins` (integer): Coins earned for this website task completion
- `task_limits` (object): Website visit task limits information (always disabled)
- `website_task` (object): Details of the completed website task
- `referral_reward` (object): Referral reward status

**Error Responses:**
- 400: Empty request body, Invalid request body, Website tasks are currently disabled, Insufficient visit duration
- 401: Invalid session or device
- 500: Database error

**Features:**
- **Visit Duration Validation**: Ensures user spent sufficient time on website
- **Daily Completion Limits**: Prevents duplicate completions of same task per day
- **Task Limits Integration**: Website tasks are unlimited
- **Referral Rewards**: Processes referral rewards if applicable
- **Transaction Safety**: Uses database transactions for data integrity

**Validation Rules:**
- Website tasks must be enabled in app settings
- Visit duration must be >= required timer duration
- Website tasks are unlimited - no completion limits

**Important Notes:**
- **Must be called after user completes website visit**
- **Visit duration must meet or exceed the required timer**
- **Website tasks are unlimited - no daily or individual task limits**
- **System randomly assigns available website tasks**
- **Default reward is 10 coins per website task completion**

**Usage Example:**
```json
// Complete website task after user visits for required duration
{
  "user_id": 1,
  "visit_duration": 35,
  "earned_coins": 10
}
```

---

### 8. Final Submit
**Endpoint:** `POST /final_submit`

**Description:** Final submission to add coins to user balance after ad completion

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "earned_coins": 10,
  "by_task": "word_puzzle",
  "created_at": "2025-01-15 10:30:00"
}
```

**Parameters:**
- `user_id` (integer, required): User ID
- `earned_coins` (integer, required): Coins earned from task completion
- `by_task` (string, optional): Task type for limits tracking
- `created_at` (string, optional): Task completion timestamp

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Final submission completed successfully",
  "data": {
    "show_ad": true,
    "total_coins": 170,
    "earned_coins": 10,
    "task_limits": {
      "enabled": true,
      "remaining_limit": 14,
      "daily_limit": 20
    },
    "referral_reward": {
      "awarded": false
    }
  },
  "new_session_token": "new_session_token_here"
}
```

**Response Data:**
- `show_ad` (boolean): Always true for final submission
- `total_coins` (integer): User's updated coin balance
- `earned_coins` (integer): Coins earned for this task completion
- `task_limits` (object): Task limits information (when by_task is provided)
- `referral_reward` (object): Referral reward status

**Task Limits Response:**
- `enabled` (boolean): Whether task limits are enabled for this task
- `remaining_limit` (integer): Remaining attempts for today (null if limits disabled or by_task not provided)
- `daily_limit` (integer): Total daily limit for this task type (null if limits disabled or by_task not provided)

**Error Responses:**
- 400: Empty request body, Invalid request body
- 401: Invalid session or device
- 500: Database error, Daily limit reached for task type

**Features:**
- **Coin Addition**: Adds earned coins to user balance
- **Task Completion Tracking**: Records task completions in database
- **Task Limits**: Enforces daily limits when by_task is provided
- **Referral Rewards**: Processes referral rewards if applicable
- **Transaction Safety**: Uses database transactions for data integrity

**Usage Example:**
```json
// Final submission after ad completion
{
  "user_id": 1,
  "earned_coins": 10,
  "by_task": "word_puzzle"
}
```

**Important Notes:**
- **Must be called after ad completion**
- **Only used when show_ad == true from complete_task.php**
- **Referral rewards are processed during coin addition**
- **Task completion records are created in database**

---

### 9. Get Task Questions
**Endpoint:** `POST /task_questions`

**Description:** Fetch questions for a specific task type

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "task_type": "word_puzzle",
  "limit": 10
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Questions fetched",
  "data": {
    "questions": [
      {
        "id": 1,
        "task_type": "word_puzzle",
        "question": "Which is correct?",
        "options": ["Aplep", "Apple", "Paple", "Leppa"],
        "correct_answer": "Apple"
      }
    ]
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid user_id, Invalid task_type, Invalid limit
- 401: Invalid session or device
- 500: Database error

---

### 10. Apply Promo Code
**Endpoint:** `POST /promocode`

**Description:** Apply a promo code to earn bonus coins

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "code": "WELCOME2025"
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Promo code applied",
  "data": {
    "coins": 100
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid promo code, Promo code expired, Usage limit reached, Promo code already used
- 401: Invalid session or device
- 500: Database error

**Features:**
- **Usage Tracking**: Prevents duplicate usage per user
- **Expiry Validation**: Checks promo code expiration dates
- **Limit Enforcement**: Respects usage limits set by admin

---

### 11. Get Redeem Categories
**Endpoint:** `POST /redeem`

**Description:** Fetch redeem categories and subcategories

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "action": "list_categories"
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Categories fetched",
  "data": {
    "categories": [
      {
        "category_id": 1,
        "category_name": "Gift Card & Payments",
        "category_image_url": "https://example.com/images/giftcard.jpg"
      }
    ]
  },
  "new_session_token": "new_session_token_here"
}
```

**Get Subcategories:**
```json
{
  "user_id": 1,
  "action": "list_subcategories",
  "category_id": 1
}
```

**Response:**
```json
{
  "status": 200,
  "message": "Subcategories fetched",
  "data": {
    "subcategories": [
      {
        "id": 1,
        "sub_category_name": "Amazon",
        "sub_category_image_url": "https://example.com/images/amazon.jpg",
        "coins": 1000,
        "cash": 10.00,
        "is_enabled": 1
      }
    ]
  },
  "new_session_token": "new_session_token_here"
}
```

---

### 12. Submit Redeem Request
**Endpoint:** `POST /redeem`

**Description:** Submit a redeem request for coins

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "action": "redeem",
  "subcategory_id": 1
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Redeem request submitted",
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid subcategory, Redeem option disabled, Insufficient coins, Invalid user
- 500: Database error

**Features:**
- **Coin Deduction**: Automatically deducts coins from user balance
- **Status Tracking**: Creates pending redeem request
- **Validation**: Checks subcategory existence and user balance

---

### 13. Get Redeem History
**Endpoint:** `POST /redeem_history`

**Description:** Fetch user's redeem request history

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Redeem history fetched",
  "data": {
    "history": [
      {
        "id": 1,
        "redeem_option_name": "Gift Card & Payments - Amazon",
        "image_url": "https://example.com/images/amazon.jpg",
        "gift_code": "GIFT123",
        "status": "pending",
        "rejection_reason": null,
        "created_at": "2025-01-15 10:30:00"
      }
    ]
  },
  "new_session_token": "new_session_token_here"
}
```

---

### 14. Get Utility Data
**Endpoint:** `POST /utility`

**Description:** Fetch utility data and app status

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "action": "check_status"
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Status fetched",
  "data": {
    "update": {
      "app_link": "https://example.com/app.apk",
      "version": "1.0.1",
      "version_code": 2,
      "changelogs": "Bug fixes and performance improvements"
    },
    "maintenance": 0,
    "vpn_allowed": 1,
    "is_blocked": 0,
    "links": {
      "how_to": "https://example.com/how-to",
      "telegram": "https://t.me/rewardapp",
      "support": "https://example.com/support"
    }
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid user_id, Invalid action
- 401: Invalid session or device

---

### 15. Daily Checkin
**Endpoint:** `POST / `

**Description:** Claim daily checkin coins

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "earned_coins": 75
}
```

**Parameters:**
- `user_id` (integer, required): User ID
- `earned_coins` (integer, optional): Custom coin amount earned by user. If not provided, uses app settings value.

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Daily checkin claimed successfully",
  "data": {
    "coins_earned": 75,
    "total_coins": 310,
    "next_checkin_available": "2025-01-16 10:30:00"
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid user_id, Already claimed today, Invalid earned_coins value
- 401: Invalid session or device
- 500: Database error

**Features:**
- **Daily Limit**: Users can only claim once per day
- **Dynamic Rewards**: App can specify custom coin amount via earned_coins parameter
- **Fallback to Settings**: If earned_coins not provided, uses admin-configured amount
- **Next Available Time**: Shows when next checkin will be available

**Usage Examples:**
```json
// App specifies custom reward (75 coins)
{
  "user_id": 1,
  "earned_coins": 75
}

// Use app settings reward (existing behavior)
{
  "user_id": 1
}
```

---

### 16. Check Daily Checkin Status
**Endpoint:** `POST /checkin_status`

**Description:** Check daily checkin status and streak information

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1
}
```

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Checkin status fetched",
  "data": {
    "can_claim": true,
    "coins_reward": 50,
    "last_claimed": "2025-01-15 10:30:00",
    "next_available": "2025-01-16 10:30:00",
    "streak_days": 3
  },
  "new_session_token": "new_session_token_here"
}
```

**Error Responses:**
- 400: Invalid user_id
- 401: Invalid session or device
- 500: Database error

**Features:**
- **Claim Status**: Shows if user can claim today
- **Reward Amount**: Displays current coin reward
- **Streak Tracking**: Shows consecutive checkin days
- **Next Available**: Shows when next checkin will be available

---

### 17. Check Task Limits
**Endpoint:** `POST /task_limits`

**Description:** Check daily task limits for the user

**Headers:**
```
Device-ID: <device_id>
Session-Token: <session_token>
Content-Type: application/json
```

**Request Body (Encrypted):**
```json
{
  "user_id": 1,
  "task_type": "word_puzzle"
}
```

**Parameters:**
- `user_id` (integer, required): User ID
- `task_type` (string, optional): Specific task type to check. If not provided, returns all task types.

**Response (Encrypted):**
```json
{
  "status": 200,
  "message": "Task limits fetched",
  "data": {
    "enabled": true,
    "limits": [
      {
        "task_type": "word_puzzle",
        "daily_limit": 20,
        "used_count": 5,
        "remaining_limit": 15
      }
    ]
  },
  "new_session_token": "new_session_token_here"
}
```

**Response Data Fields:**
- `enabled` (boolean): Whether task limits are globally enabled
- `limits` (array): Array of task limit objects
  - `task_type` (string): Task type name
  - `daily_limit` (integer): Total daily limit for this task type
  - `used_count` (integer): Number of times used today
  - `remaining_limit` (integer): Remaining attempts for today
  - `note` (string, optional): Additional information (e.g., "Task type not limited")

**Error Responses:**
- 400: Invalid user_id
- 401: Invalid session or device
- 500: Database error

**Features:**
- **Check All Limits**: Get limits for all task types when task_type not provided
- **Check Specific Limit**: Get limit for specific task type
- **Real-time Data**: Shows current usage and remaining attempts
- **Non-existent Tasks**: Handles task types not in configuration
- **Global Control**: Respects global task limits enable/disable setting

**Usage Examples:**
```json
// Check all task limits
{
  "user_id": 1
}

// Check specific task limit
{
  "user_id": 1,
  "task_type": "word_puzzle"
}

// Check non-existent task type
{
  "user_id": 1,
  "task_type": "custom_task"
}
```

**Response Examples:**
```json
// All task limits
{
  "enabled": true,
  "limits": [
    {
      "task_type": "word_puzzle",
      "daily_limit": 20,
      "used_count": 5,
      "remaining_limit": 15
    },
    {
      "task_type": "alphabet_puzzle",
      "daily_limit": 15,
      "used_count": 3,
      "remaining_limit": 12
    }
  ]
}

// Non-existent task type
{
  "enabled": true,
  "limits": [
    {
      "task_type": "custom_task",
      "daily_limit": null,
      "used_count": 0,
      "remaining_limit": null,
      "note": "Task type not limited"
    }
  ]
}

// Task limits disabled
{
  "enabled": false,
  "limits": []
}
```

---

## Gamezop API Integration

### Overview
The Gamezop API integration provides access to a curated collection of games through the Gamezop platform. This API allows users to search, filter, and retrieve game information for integration into the quiz app.

### Base URL
```
https://snaplimit.com/api/user/gamezop_api.php
```

### Authentication
- **No authentication required** - This is a public API endpoint
- **Rate Limit**: Standard rate limiting applies
- **Data Source**: Fetches fresh data from Gamezop API (no caching)

### Endpoints

#### 1. Welcome/Default Endpoint
**Endpoint:** `GET /gamezop_api.php`

**Description:** Returns welcome message and available endpoints

**Response:**
```json
{
  "message": "🎮 Welcome to AllGame API",
  "endpoints": {
    "Search games": "/api.php?action=search&query=pop",
    "Category games": "/api.php?action=category&name=Puzzle%20%26%20Logic",
    "Tag games": "/api.php?action=tag&tag=arcade",
    "All games": "/api.php?action=all",
    "All categories": "/api.php?action=list-categories",
    "All tags": "/api.php?action=list-tags",
    "Game details by code": "/api.php?action=game-details&code=SJX7TGkDq1X"
  }
}
```

#### 2. Search Games
**Endpoint:** `GET /gamezop_api.php?action=search&query={search_term}`

**Description:** Search games by name or description

**Parameters:**
- `action` (string, required): Must be "search"
- `query` (string, required): Search term to find games

**Example Request:**
```
GET /gamezop_api.php?action=search&query=pop
```

**Response:**
```json
[
  {
    "code": "SJX7TGkDq1X",
    "name": {
      "en": "Pop the Balloons"
    },
    "description": {
      "en": "Pop colorful balloons in this fun arcade game"
    },
    "categories": {
      "en": ["Arcade", "Puzzle"]
    },
    "tags": {
      "en": ["arcade", "balloons", "fun"]
    }
  }
]
```

#### 3. Get Games by Category
**Endpoint:** `GET /gamezop_api.php?action=category&name={category_name}`

**Description:** Filter games by category

**Parameters:**
- `action` (string, required): Must be "category"
- `name` (string, required): Category name (URL encoded)

**Example Request:**
```
GET /gamezop_api.php?action=category&name=Puzzle%20%26%20Logic
```

**Response:**
```json
[
  {
    "code": "ABC123",
    "name": {
      "en": "Logic Puzzle"
    },
    "description": {
      "en": "Solve challenging logic puzzles"
    },
    "categories": {
      "en": ["Puzzle & Logic"]
    },
    "tags": {
      "en": ["puzzle", "logic", "brain"]
    }
  }
]
```

#### 4. Get Games by Tag
**Endpoint:** `GET /gamezop_api.php?action=tag&tag={tag_name}`

**Description:** Filter games by tag

**Parameters:**
- `action` (string, required): Must be "tag"
- `tag` (string, required): Tag name

**Example Request:**
```
GET /gamezop_api.php?action=tag&tag=arcade
```

**Response:**
```json
[
  {
    "code": "XYZ789",
    "name": {
      "en": "Arcade Shooter"
    },
    "description": {
      "en": "Fast-paced arcade shooting game"
    },
    "categories": {
      "en": ["Arcade", "Action"]
    },
    "tags": {
      "en": ["arcade", "shooter", "action"]
    }
  }
]
```

#### 5. Get All Games
**Endpoint:** `GET /gamezop_api.php?action=all`

**Description:** Retrieve all available games

**Parameters:**
- `action` (string, required): Must be "all"

**Example Request:**
```
GET /gamezop_api.php?action=all
```

**Response:**
```json
[
  {
    "code": "SJX7TGkDq1X",
    "name": {
      "en": "Pop the Balloons"
    },
    "description": {
      "en": "Pop colorful balloons in this fun arcade game"
    },
    "categories": {
      "en": ["Arcade", "Puzzle"]
    },
    "tags": {
      "en": ["arcade", "balloons", "fun"]
    }
  },
  {
    "code": "ABC123",
    "name": {
      "en": "Logic Puzzle"
    },
    "description": {
      "en": "Solve challenging logic puzzles"
    },
    "categories": {
      "en": ["Puzzle & Logic"]
    },
    "tags": {
      "en": ["puzzle", "logic", "brain"]
    }
  }
]
```

#### 6. List All Categories
**Endpoint:** `GET /gamezop_api.php?action=list-categories`

**Description:** Get list of all available categories

**Parameters:**
- `action` (string, required): Must be "list-categories"

**Example Request:**
```
GET /gamezop_api.php?action=list-categories
```

**Response:**
```json
[
  "Arcade",
  "Puzzle & Logic",
  "Action",
  "Adventure",
  "Racing",
  "Sports"
]
```

#### 7. List All Tags
**Endpoint:** `GET /gamezop_api.php?action=list-tags`

**Description:** Get list of all available tags

**Parameters:**
- `action` (string, required): Must be "list-tags"

**Example Request:**
```
GET /gamezop_api.php?action=list-tags
```

**Response:**
```json
[
  "arcade",
  "puzzle",
  "logic",
  "action",
  "adventure",
  "racing",
  "sports",
  "balloons",
  "fun",
  "brain"
]
```

#### 8. Get Game Details by Code
**Endpoint:** `GET /gamezop_api.php?action=game-details&code={game_code}`

**Description:** Get detailed information about a specific game

**Parameters:**
- `action` (string, required): Must be "game-details"
- `code` (string, required): Unique game code

**Example Request:**
```
GET /gamezop_api.php?action=game-details&code=SJX7TGkDq1X
```

**Response:**
```json
{
  "code": "SJX7TGkDq1X",
  "name": {
    "en": "Pop the Balloons"
  },
  "description": {
    "en": "Pop colorful balloons in this fun arcade game"
  },
  "categories": {
    "en": ["Arcade", "Puzzle"]
  },
  "tags": {
    "en": ["arcade", "balloons", "fun"]
  }
}
```

**Error Response (Game Not Found):**
```json
{
  "error": "Game not found with code: INVALID_CODE"
}
```

### Game Object Structure
```json
{
  "code": "string",
  "name": {
    "en": "string"
  },
  "description": {
    "en": "string"
  },
  "categories": {
    "en": ["string"]
  },
  "tags": {
    "en": ["string"]
  }
}
```

### Features
- **Real-time Data**: Always fetches fresh data from Gamezop API
- **Case-insensitive Search**: Search queries are case-insensitive
- **Multi-language Support**: Games support multiple languages (currently English)
- **Flexible Filtering**: Filter by category, tag, or search terms
- **Comprehensive Data**: Access to game codes, names, descriptions, categories, and tags

### Usage Examples
```javascript
// Search for puzzle games
fetch('/gamezop_api.php?action=search&query=puzzle')

// Get all arcade games
fetch('/gamezop_api.php?action=category&name=Arcade')

// Get games with "fun" tag
fetch('/gamezop_api.php?action=tag&tag=fun')

// Get specific game details
fetch('/gamezop_api.php?action=game-details&code=SJX7TGkDq1X')

// Get all available categories
fetch('/gamezop_api.php?action=list-categories')

// Get all available tags
fetch('/gamezop_api.php?action=list-tags')
```

### Error Handling
- **API Unavailable**: Returns error message if Gamezop API is unreachable
- **Invalid Game Code**: Returns error message for non-existent game codes
- **Empty Results**: Returns empty array for searches with no matches
- **Missing Parameters**: Returns default welcome message

---

## Error Codes

| Code | Description |
|------|-------------|
| 200 | Success |
| 400 | Bad Request - Invalid input or business logic error |
| 401 | Unauthorized - Invalid session or device |
| 404 | Not Found - Resource not found |
| 405 | Method Not Allowed - Wrong HTTP method |
| 429 | Too Many Requests - Rate limit exceeded |
| 500 | Internal Server Error - Database or server error |

## Rate Limiting
- **Limit**: 100 requests per hour per user
- **Headers**: Rate limit information is not exposed in responses
- **Penalty**: Temporary block for exceeding limits

## Security Features
- **Request Encryption**: All requests are encrypted using custom encryption system
- **Response Encryption**: All responses are encrypted
- **Session Management**: Automatic session token refresh
- **Device Binding**: Sessions are bound to specific device IDs
- **Input Validation**: Comprehensive validation for all inputs
- **One Device Policy**: Each device can only be associated with one account

## Data Types

### User Object
```json
{
  "id": 1,
  "email": "user@example.com",
  "device_id": "device123",
  "coins": 100,
  "refercode": "userxIrP",
  "referred_by": null,
  "created_at": "2025-01-15 10:30:00",
  "is_blocked": 0
}
```

### Task Object
```json
{
  "id": 1,
  "task_type": "word_puzzle",
  "reward_coins": 10,
  "is_enabled": 1
}
```

### Redeem Object
```json
{
  "id": 1,
  "user_id": 1,
  "subcategory_id": 1,
  "image_url": "https://example.com/proof.jpg",
  "gift_code": "GIFT123",
  "status": "pending",
  "rejection_reason": null,
  "created_at": "2025-01-15 10:30:00"
}
```

### Daily Checkin Object
```json
{
  "id": 1,
  "user_id": 1,
  "coins_earned": 50,
  "claimed_at": "2025-01-15 10:30:00"
}
```

## Best Practices
1. **Always handle errors gracefully** - Check status codes and error messages
2. **Store session tokens securely** - Use secure storage methods
3. **Implement retry logic** - For network failures with exponential backoff
4. **Validate responses** - Ensure data integrity before using
5. **Handle rate limits** - Implement proper backoff strategies
6. **Use HTTPS** - Always use secure connections in production
7. **Respect device policy** - One device per account only
8. **Handle encryption** - Properly encrypt/decrypt all requests and responses
9. **Check daily checkin status** - Use checkin_status endpoint before attempting to claim
10. **Track user engagement** - Monitor daily checkin streaks for user retention 